﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using UnityEngine;

//ColoursManager is a singleton class responsible for managing all parsed colours from Colours.xml
public class ColoursManager : MonoBehaviour
{
    public Dictionary<string, ColourEntry> Colours { get; private set; } = new Dictionary<string, ColourEntry>();   //Colours on their own, with an ID key and ColourEntry value
    public Dictionary<string, ColourGroup> ColourGroups { get; private set; } = new Dictionary<string, ColourGroup>();  //ColourGroup, with an ID key and ColourEntry value

    //Singleton
    private static ColoursManager _Instance;
    public static ColoursManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<ColoursManager>();
            }

            return _Instance;
        }
    }

    public bool Initialize()
    {
        try
        {
            return ParseColoursXML();
        }

        catch (Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }

    public List<ColourEntry> GetColoursWithTag(string tag)
    {
        List<ColourEntry> retList = new List<ColourEntry>();

        //Loop through all colours, add colours to the list that have the passed tag
        foreach (ColourEntry colourEntry in Colours.Values)
        {
            if (colourEntry.Tags.Contains(tag))
            {
                retList.Add(colourEntry);
            }
        }

        return retList;
    }
    private bool ParseColoursXML()
    {
        try
        {
            //Load our XML
            TextAsset xmlText = (TextAsset)Resources.Load("Colours");
            XDocument coloursXML = XDocument.Parse(xmlText.text);

            var coloursXMLList = (from ele in coloursXML.Descendants("Colour")
                                  select ele).ToList();    //Get all the colour elements first

            for (int i = 0; i < coloursXMLList.Count; i++)
            {
                XElement colourElem = coloursXMLList[i];
                if (colourElem.Attribute("Key") != null)
                {
                    string thisColourKey = colourElem.Attribute("Key").Value;   //We've got a key

                    if (!Colours.ContainsKey(thisColourKey))
                    {
                        Color thisParsedColour;

                        if (ColorUtility.TryParseHtmlString(colourElem.Value, out thisParsedColour))    //This key is unique, so let's parse the colour
                        {
                            ColourEntry thisColourEntry;

                            if (colourElem.Attribute("Tags") != null)   //We've got tags, call the appropriate constructor
                            {
                                string thisColourTagAttributeValue = colourElem.Attribute("Tags").Value;
                                List<string> thisColourTags = thisColourTagAttributeValue.Split('|').ToList();
                                thisColourEntry = new ColourEntry(thisColourKey, thisColourTags, thisParsedColour);
                            }

                            else    //We don't have tags, call the appropriate constructor
                            {
                                thisColourEntry = new ColourEntry(thisColourKey, thisParsedColour);
                            }

                            Colours.Add(thisColourKey, thisColourEntry);
                        }

                        else
                        {
                            Debug.LogError(string.Format("ERROR: Colour key {0} has an invalid colour HTML string.", thisColourKey));
                        }
                    }

                    else
                    {
                        Debug.LogError(string.Format("ERROR: Colour key {0} is a duplicate.", thisColourKey));
                    }
                }

                else
                {
                    Debug.LogError(string.Format("ERROR: Colour index {0} has no key.", i));
                }
            }

            var groupsXMLList = (from ele in coloursXML.Descendants("ColourGroup")
                                 select ele).ToList();    //Now handle the groups

            for (int i = 0; i < groupsXMLList.Count; i++)
            {
                XElement groupElem = groupsXMLList[i];

                ColourGroup colourGroup = new ColourGroup();
                if (colourGroup.Parse(groupElem))   //Get the group, let's try parsing it
                {
                    if (!ColourGroups.ContainsKey(colourGroup.Name))
                    {
                        ColourGroups[colourGroup.Name] = colourGroup;   //Not a duplicate so let's add it
                    }

                    else
                    {
                        Debug.LogError(string.Format("ERROR: Colour group key {0} is a duplicate.", colourGroup.Name));
                    }
                }

                else
                {
                    Debug.LogError(string.Format("ERROR: Failed to parse colour group index {0}. Perhaps the XML is malformed?", i));
                }
            }

            return true;
        }

        catch (Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }
}
